################################################################
#
#   Bell Runner - Level Editor Screen
#
################################################################

from __future__ import division
import os
from pygame import Rect
from pygame.transform import rotozoom
from albow.widget import Widget
from albow.controls import Label, Row, Column
from albow.dialogs import Dialog
from albow.palette_view import PaletteView
from albow.resource import resource_path, get_image, get_font
from screen import BRScreen
from level_editor_view import LevelEditorView
from game import game
import level
from level import Background, Foreground, Platform, Wall, get_item_image
from bell import Bell
from exit import Exit

class EditorScreen(BRScreen):

	def __init__(self, shell):
		BRScreen.__init__(self, shell)
		self.level_view = LevelEditorView()
		tp = ToolPalette()
		ci = CurrentItemView()
		bgm = ItemMenu(Background, (66, 66), 8, 8, 0.5)
		plm = ItemMenu(Platform, (66, 18), 4, 8, 0.5)
		wam = ItemMenu(Wall, (18, 66), 8, 4, 0.5)
		blm = ItemMenu(Bell, (66, 66), 4, 4, 0.5)
		exm = ItemMenu(Exit, (66, 66), 4, 4, 0.5)
		fgm = ItemMenu(Foreground, (66, 66), 8, 8, 0.5)
		bgb = PopupButton("Bg", bgm)
		fgb = PopupButton("Fg", fgm)
		plb = PopupButton("Platforms", plm)
		wab = PopupButton("Walls", wam)
		blb = PopupButton("Bells", blm)
		exb = PopupButton("Exits", exm)
		toolbar = Row([tp, ci, bgb, fgb, exb, plb, wab, blb], align = 'c')
		toolbar.midleft = (0, 588)
		self.add(self.level_view)
		self.add(toolbar)
		self.level_view.focus()
	
	def key_down(self, e):
		if e.cmd:
			self.shell.edit_menu_screen.key_down(e)
		else:
			BRScreen.key_down(self, e)
	
	def escape_key(self):
		self.shell.show_edit_menu_screen()


class CurrentItemView(Widget):

	def __init__(self):
		Widget.__init__(self, Rect(0, 0, 24, 24))
	
	def draw(self, surf):
		kind = game.current_item_kind
		name = game.current_item_name
		if kind and name:
			image = get_item_image(kind, name)
			if image:
				r = Rect(1, 1, 22, 22)
				blit_scaled(surf, image, r)


class PopupButton(Label):

	font = get_font(18, "Vera.ttf")

	def __init__(self, title, client):
		Label.__init__(self, title)
		self.dlog = PopupDialog(client)
	
	def mouse_down(self, e):
		self.dlog.bottomleft = self.parent.local_to_global(self.topleft)
		self.dlog.present(centered = False)


class PopupDialog(Dialog):

	#border_color = (128, 128, 128)
	border_width = 0

	def __init__(self, client):
		r = Rect(0, 0, client.width + 10, client.height + 10)
		Widget.__init__(self, r)
		self.client = client
		client.center = (self.width/2, self.height/2)
		self.add(client)
	
	def mouse_down(self, e):
		self.dismiss()


class ItemMenu(PaletteView):

	scale = None

	def __init__(self, kind, cell_size = (32, 32), nrows = 16, ncols = 16,
			scale = None):
		PaletteView.__init__(self, cell_size, nrows, ncols, scrolling = True)
		self.scale = scale
		self.kind = kind
		dir = resource_path("images/%s" % kind.resource_prefix)
		names = os.listdir(dir)
		names.sort()
		suffix = kind.filename_suffix
		self.names = [n for n in names if n.endswith(suffix)]
	
	def num_items(self):
		return len(self.names)
	
	def draw_item(self, surf, i, rect):
		image = get_item_image(self.kind, self.names[i])
		r = rect.inflate((-2, -2))
		s = self.scale
		w, h = image.get_size()
		if w * s > r.width or h * s > r.height:
			s = scale_to_fit((w, h), r.size)
		blit_scaled(surf, image, r, s)
	
	def click_item(self, i, e):
		game.current_item_kind = self.kind
		game.current_item_name = self.names[i]
		game.current_tool = 'add'
		self.parent.dismiss()
	
	def item_is_selected(self, i):
		return game.current_item_kind == self.kind \
			and game.current_item_name == self.names[i]


class ToolPalette(PaletteView):

	tool_names = ['select', 'add'] #, 'remove']
	image_names = ["select.png", "add.png"] #, "remove.png"]

	def __init__(self):
		PaletteView.__init__(self, (24, 24), 1, 3)
		self.images = [get_image("tools/%s" % name) for name in self.image_names]
	
	def num_items(self):
		return len(self.tool_names)
	
	def draw_item(self, surf, i, rect):
		image = self.images[i]
		r = image.get_rect()
		r.center = rect.center
		surf.blit(image, r)
	
	def click_item(self, i, e):
		game.current_tool = self.tool_names[i]
	
	def item_is_selected(self, i):
		return game.current_tool == self.tool_names[i]


def blit_scaled(dst_surf, src_surf, dst_rect, scale = None):
	if scale is None:
		scale = scale_to_fit(src_surf.get_size(), dst_rect.size)
	buf = rotozoom(src_surf, 0, scale)
	r = buf.get_rect()
	r.center = dst_rect.center
	dst_surf.blit(buf, r)

def scale_to_fit(src_size, dst_size):
	src_width, src_height = src_size
	dst_width, dst_height = dst_size
	if src_width > src_height:
		scale = dst_width / src_width
	else:
		scale = dst_height / src_height
	return scale
