#
#   Demonstration application for Albow
#
#   Greg Ewing, October 2006
#

screen_size = (640, 480)
flags = 0
frame_time = 50 # ms

import sys

import pygame
pygame.init()
from pygame.color import Color
from pygame.locals import *

from albow.widget import Widget
from albow.controls import Label, Button, TextField
from albow.shell import Shell, TextScreen
from albow.resource import get_font
from albow.grid_view import GridView
from albow.palette_view import PaletteView

#--------------------------------------------------------------------------------
#
#    Text Field
#
#--------------------------------------------------------------------------------

class DemoTextFieldsScreen(Widget):

	def __init__(self, shell):
		Widget.__init__(self, shell.rect)
		self.fld1 = self.add_field("Name", 200)
		self.fld2 = self.add_field("Race", 250)
		btn = Button("OK", action = self.ok)
		btn.rect.midtop = (320, 300)
		self.add(btn)
		out = Label("")
		out.rect.width = 400
		out.rect.topleft = (200, 350)
		self.out = out
		self.add(out)
		btn = Button("Menu", action = self.go_back)
		btn.rect.midtop = (320, 400)
		self.add(btn)
	
	def add_field(self, label, pos):
		lbl = Label(label)
		lbl.rect.topleft = (200, pos)
		self.add(lbl)
		fld = TextField(300)
		fld.rect.topleft = (250, pos)
		fld.enter_action = self.ok
		self.add(fld)
		return fld
	
	def ok(self):
		self.out.text = "You are a %s called %s." % (self.fld2.text, self.fld1.text)
	
	def go_back(self):
		self.parent.show_menu()

#--------------------------------------------------------------------------------
#
#    Buttons
#
#--------------------------------------------------------------------------------

class MenuScreen(Widget):

	def __init__(self, shell):
		Widget.__init__(self, shell.rect)
		self.shell = shell
		f1 = get_font(24)
		title = Label("Albow Demo", f1)
		title.rect.center = (320, 100)
		self.add(title)
		self.add_screen_button("Text Screen", 150, shell.text_screen)
		self.add_screen_button("Text Fields", 175, shell.fields_screen)
		self.add_screen_button("Timing", 200, shell.anim_screen)
		self.add_screen_button("Grid View", 225, shell.grid_screen)
		self.add_screen_button("Palette View", 250, shell.palette_screen)
		self.add_button("Quit", 400, self.quit)
	
	def add_screen_button(self, text, pos, screen):
		self.add_button(text, pos, lambda: self.shell.show_screen(screen))
	
	def add_button(self, text, pos, action):
		button = Button(text, action = action)
		button.rect.center = (320, pos)
		self.add(button)
	
	def show_text_screen(self):
		self.shell.show_screen(self.text_screen)
	
	def show_fields_screen(self):
		self.shell.show_screen(self.fields_screen)
		self.fields_screen.fld1.focus()
	
	def show_anim_screen(self):
		self.shell.show_screen(self.anim_screen)
	
	def quit(self):
		sys.exit(0)

#--------------------------------------------------------------------------------
#
#    Animation
#
#--------------------------------------------------------------------------------

class DemoAnimScreen(Widget):

	def __init__(self, shell):
		r = shell.rect.inflate(-100, -100)
		Widget.__init__(self, r)
		w, h = r.size
		self.points = [[100, 50], [w - 50, 100], [50, h - 50]]
		from random import randint
		def randv():
			return randint(-5, 5)
		self.velocities = [[randv(), randv()] for i in range(len(self.points))]
		btn = Button("Menu", action = self.go_back)
		btn.rect.center = (w/2, h - 20)
		self.add(btn)
	
	def draw(self, surface):
		from pygame.draw import polygon
		polygon(surface, (128, 200, 255), self.points)
		polygon(surface, (255, 128, 0), self.points, 5)
	
	def begin_frame(self):
		r = self.rect
		w, h = r.size
		for p, v in zip(self.points, self.velocities):
			p[0] += v[0]
			p[1] += v[1]
			if not 0 <= p[0] <= w:
				v[0] = -v[0]
			if not 0 <= p[1] <= h:
				v[1] = -v[1]

	def go_back(self):
		self.parent.show_menu()

#--------------------------------------------------------------------------------
#
#   Grid View
#
#--------------------------------------------------------------------------------

class DemoGridView(GridView):

	info = [
		[("red", "r3d"), ("green", "gr33n"), ("blue", "blu3")],
		[("cyan", "cy4n"), ("magenta", "m4g3nt4"), ("yellow", "y3ll0w")]
	]

	def __init__(self):
		GridView.__init__(self, (30, 30), 2, 3)
	
	def num_rows(self):
		return 2
	
	def num_cols(self):
		return 3
	
	def draw_cell(self, surface, row, col, rect):
		color = Color(self.info[row][col][0])
		surface.fill(color, rect)
	
	def click_cell(self, row, col, event):
		self.output.text = self.info[row][col][1]

#--------------------------------------------------------------------------------

class DemoGridViewScreen(Widget):

	def __init__(self, shell):
		Widget.__init__(self, shell.rect)
		w, h = self.rect.size
		grid = DemoGridView()
		grid.rect.center = (w/2, h/2)
		self.add(grid)
		lbl = Label("")
		lbl.rect.topleft = (grid.rect.left, grid.rect.bottom + 10)
		self.add(lbl)
		grid.output = lbl
		btn = Button("Menu", action = self.go_back)
		btn.rect.center = (w/2, h - 50)
		self.add(btn)

	def go_back(self):
		self.parent.show_menu()

#--------------------------------------------------------------------------------
#
#    Palette View
#
#--------------------------------------------------------------------------------

class DemoPaletteView(PaletteView):

	info = ["red", "green", "blue", "cyan", "magenta", "yellow"]
	
	sel_color = Color("white")
	sel_width = 5

	def __init__(self):
		PaletteView.__init__(self, (30, 30), 2, 2, scrolling = True)
		self.selection = None
	
	def num_items(self):
		return len(self.info)
	
	def draw_item(self, surface, item_no, rect):
		color = Color(self.info[item_no])
		surface.fill(color, rect)
	
	def click_item(self, item_no, event):
		self.selection = item_no
	
	def item_is_selected(self, item_no):
		return self.selection == item_no

#--------------------------------------------------------------------------------

class DemoPaletteViewScreen(Widget):

	def __init__(self, shell):
		Widget.__init__(self, shell.rect)
		w, h = self.rect.size
		grid = DemoPaletteView()
		grid.rect.center = (w/2, h/2)
		self.add(grid)
		btn = Button("Menu", action = self.go_back)
		btn.rect.center = (w/2, h - 50)
		self.add(btn)

	def go_back(self):
		self.parent.show_menu()
	
#--------------------------------------------------------------------------------
#
#    Shell
#
#--------------------------------------------------------------------------------

class DemoShell(Shell):

	def __init__(self, display):
		Shell.__init__(self, display)
		self.text_screen = TextScreen("demo_text.txt")
		self.fields_screen = DemoTextFieldsScreen(self)
		self.anim_screen = DemoAnimScreen(self)
		self.grid_screen = DemoGridViewScreen(self)
		self.palette_screen = DemoPaletteViewScreen(self)
		self.menu_screen = MenuScreen(self)
		self.set_timer(frame_time)
		self.show_menu()
	
	def show_menu(self):
		self.show_screen(self.menu_screen)
	
	def begin_frame(self):
		self.anim_screen.begin_frame()

def main():
	display = pygame.display.set_mode(screen_size, flags)
	shell = DemoShell(display)
	shell.run()

main()
