from pygame import Rect, draw
from widget import Widget
import resource

class Label(Widget):

	fg_color = (255, 255, 255)
	
	def __init__(self, text, font = resource.default_font):
		Widget.__init__(self, Rect((0, 0), font.size(text)))
		self.text = text
		self.font = font

	def draw(self, surface):
		self.draw_with(surface, self.fg_color)
	
	def draw_with(self, surface, fg):
		image = self.font.render(self.text, True, fg)
		surface.blit(image, (0, 0))


class Button(Label):

	fg_color = (255, 255, 0)
	highlight_color = (255, 0, 0)

	def __init__(self, text, font = None, action = None):
		font = font or resource.get_font(15, "VeraBd.ttf")
		Label.__init__(self, text, font)
		if action:
			self.action = action
		self.highlighted = False
		self.enabled = True
	
	def draw(self, surface):
		if self.enabled:
			if self.highlighted:
				color = self.highlight_color
			else:
				color = self.fg_color
			self.draw_with(surface, color)
	
	def mouse_down(self, event):
		if self.enabled:
			self.highlighted = True
	
	def mouse_up(self, event):
		if self.highlighted:
			self.highlighted = False
			self.call_handler('action')
	
	def action(self):
		pass


class TextField(Widget):

	fg_color = (0, 255, 0)

	def __init__(self, width, font = None):
		font = font or resource.default_font
		Widget.__init__(self, Rect(0, 0, width, font.get_linesize()))
		self.font = font
		self.text = ""
	
	def get_text(self):
		return self.text
	
	def set_text(self, text):
		self.text = text
	
	def draw(self, surface):
		fg = self.fg_color
		image = self.font.render(self.text, True, fg)
		surface.blit(image, (0, 0))
		if self.has_focus():
			r = image.get_rect()
			draw.line(surface, fg, r.topright, r.bottomright)
	
	def key_down(self, event):
		c = event.unicode
		if c <= "\x7f":
			if c == "\x08" or c == "\x7f":
				self.text = self.text[:-1]
				return
			elif c == "\r" or c == "\x03":
				if self.call_handler('enter_action'):
					return
			elif c == "\x1b":
				if self.call_handler('escape_action'):
					return
			elif c >= "\x20":
				self.text = self.text + c
				return
		self.call_parent_handler('key_down', event)

	def mouse_down(self, event):
		self.focus()
